patches-own [
  elevation
]

globals [
  water-height    ;; how high the floods are currently
  raise-water?    ;; true or false, are we ready for the water to go higher?
  ;; The rest of the variables are used only for coloring.
  flood-1-color flood-2-color
  ocean-color divide-color
  initial-ground-color flooded-ground-colors
]

;;;
;;; SETUP PROCEDURES
;;;

to setup
  clear-all
  set-default-shape turtles "circle"
  setup-colors
  setup-elevations
  setup-floods
  color-world
  reset-ticks
end

to setup-elevations
  ;; Here we're using FOREACH to walk down two lists simultaneously:
  ;; the list of patches, and the list of elevations.
  (foreach (sort patches) elevation-data [ [the-patch the-elevation] ->
    ask the-patch [ set elevation the-elevation ]
  ])
end

to setup-floods
  ask patches [
    if elevation = -1 [
      sprout 1 [ set color flood-1-color ]
    ]
    if elevation = -2 [
      sprout 1 [ set color flood-2-color ]
    ]
  ]
end

;;;
;;; MAIN PROCEDURES
;;;

to go
  if not any? turtles [ stop ]
  set raise-water? true
  ask turtles [ flood ]
  ask turtles [ check-for-divide ]
  ;;; if raise-water? didn't get set to false when FLOOD happened,
  ;;; there won't be more flooding at the current water-height, so raise it
  if raise-water? [
    ;; raising by 5 is less accurate than raising by 1, but it's faster
    set water-height water-height + 5
  ]
  tick
end

to flood  ;; turtle procedure
  let my-color color
  let unflooded-neighbors neighbors4 with [shade-of? pcolor initial-ground-color and
                                           not any? turtles-here with [color = my-color]]
  if not any? unflooded-neighbors [
    recolor-patch
    ;; we won't do any more flooding from this patch
    die
  ]
  ask unflooded-neighbors with [elevation < water-height] [
    sprout 1 [
      set color my-color
      set raise-water? false
    ]
  ]
end

;;; if the two floods are colliding, we must be on the divide
to check-for-divide  ;; turtle procedure
  if any? other turtles-here [
    set pcolor divide-color
    ask turtles-here [ die ]  ;; turtles-here includes me, so I die too
  ]
end

;;;
;;; COLOR PROCEDURES
;;;

to setup-colors
  set flood-1-color yellow + 1
  set flood-2-color orange + 1
  set ocean-color black
  set divide-color red
  set initial-ground-color brown
  set flooded-ground-colors [magenta blue sky turquoise lime green]
end

to color-world
  ask patches [
    ifelse elevation < 0 [    ;; check: is ocean?
      set pcolor ocean-color
    ] [
      set pcolor scale-color initial-ground-color elevation -250 max-elevation
    ]
  ]
end

to recolor-patch  ;; turtle procedure
  ;; while staying within the bounds of the list,
  ;; find the (water-height / 150)th element of the flooded-ground-colors
  let current-flooded-ground-color item (max list 0
                                                 (min list (length flooded-ground-colors - 1)
                                                           (floor (water-height / 150))))
                                        flooded-ground-colors
  set pcolor scale-color current-flooded-ground-color elevation -250 max-elevation ; -250 makes things lighter than the ocean
end

;;;
;;; THE MAP
;;;

to-report max-elevation
  report 3000
end

to-report elevation-data
  ;; -3 is ocean, -1 is the west coast, -2 is the east coast
  report [
    -3 -3 -3 -3 -1 32 92 145 190 258 348 426 493 544 580 603 614 610 589 572 558 571 610 627 621 650 714 762 793 831 876 878 838 817 816 861 951 1008 1032 1063 1100 1163 1250 1285 1266 1239 1201 1149 1081 1032 1001 958 904 849 792 742 698 628 533 486 489 490 489 467 424 335 201 124 106 101 109 119 130 144 161 172 179 180 175 174 176 207 267 292 280 274 274 275 275 275 274 274 274 274 274 281 295 339 413 418 354 287 215 169 148 120 86 45 -3 -3 -3 -3 -3 -3 3 2 -3 -3 -3 -3 -3 -3 -3 8 27 38 39 26 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 7 22 25 16 4 -3 -3 -3 -3 44 90 138 177 208 183 102 85 131 138 105 55 -3 -3 -3 -3 114 206 270 305 310 255 141 60 13 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -1 15 60 102 142 204 287 354 405 456 509 545 563 567 557 542 523 532 570 602 629 679 752 800 823 854 891 890 849 824 815 846 917 973 1015 1045 1062 1092 1134 1134 1090 1052 1020 976 919 883 867 836 790 753 724 691 653 595 518 483 491 498 505 512 519 437 266 161 123 107 113 124 140 155 170 180 185 187 184 182 179 207 264 289 279 276 278 279 280 279 276 276 278 277 275 279 291 329 392 399 348 292 231 191 173 146 110 72 30 8 5 6 10 16 22 25 24 19 12 3 -3 -3 17 40 65 85 100 97 75 60 53 41 24 12 5 8 23 22 6 2 11 21 31 32 24 14 2 4 19 46 84 125 168 204 232 217 157 146 184 184 148 90 11 -3 -3 43 153 231 277 303 310 265 167 87 24 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -1 -1 21 50 99 168 213 234 286 371 430 464 485 494 484 455 458 493 556 645 735 825 874 881 897 921 912 871 837 813 817 850 906 983 1010 987 954 910 842 749 691 669 640 606 595 607 599 570 568 593 592 566 531 490 477 494 514 536 600 704 635 393 233 155 118 122 136 158 176 189 196 198 201 203 198 186 206 259 282 278 278 284 288 290 287 281 279 284 283 275 276 283 309 353 362 337 304 263 235 222 196 157 123 93 75 70 64 57 55 59 69 84 90 85 67 36 35 66 101 138 178 219 233 220 211 204 185 155 134 124 123 130 111 65 44 47 49 48 45 40 35 30 52 99 136 162 193 228 257 279 281 264 265 286 274 230 158 56 27 70 137 227 278 290 301 310 283 218 139 47 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -1 8 40 83 106 111 153 232 292 332 358 371 368 350 361 400 465 554 635 707 745 748 760 780 772 737 709 689 701 744 791 843 840 785 722 651 569 476 422 408 392 375 378 401 416 423 457 520 565 594 600 585 566 542 533 539 614 757 720 504 336 216 152 144 149 168 181 189 192 193 193 193 188 176 194 240 265 269 276 287 291 286 278 265 261 265 261 249 242 241 257 288 300 292 286 282 280 280 265 236 212 195 185 182 176 168 165 167 175 188 192 186 165 129 118 134 163 205 245 286 302 296 290 285 269 241 222 213 205 199 168 113 86 85 83 79 75 71 70 73 107 172 215 235 253 270 286 301 315 329 338 344 328 291 223 125 88 114 155 212 240 239 241 247 227 181 115 28 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -1 -1 -1 -1 15 25 28 31 33 53 91 130 168 189 192 199 209 240 292 333 363 389 409 425 436 453 477 478 457 448 451 502 600 634 601 544 461 400 360 315 266 238 231 226 222 229 245 284 344 417 501 605 730 794 796 743 633 557 517 560 686 699 601 469 304 207 177 165 170 172 171 170 169 165 158 154 151 170 210 238 254 270 287 287 270 252 232 222 222 213 196 181 169 175 199 213 216 242 291 327 348 353 344 337 333 334 339 341 341 341 341 338 333 324 312 294 269 244 220 227 264 290 304 309 305 302 300 295 286 279 274 258 230 194 149 127 125 124 122 120 116 118 128 168 239 284 304 308 297 294 300 320 354 367 359 347 331 287 217 160 117 101 112 120 126 127 123 101 62 18 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -1 13 28 40 51 61 67 69 65 56 55 60 80 113 135 146 160 176 194 215 225 225 234 249 276 312 335 346 344 331 337 363 423 518 530 460 388 312 285 306 308 291 284 285 279 265 262 271 310 379 453 534 660 833 924 936 877 748 660 614 631 714 730 680 590 459 386 371 367 374 366 342 328 323 298 253 224 211 211 222 238 260 274 282 277 257 240 225 213 204 194 183 171 157 160 180 194 202 242 313 360 383 395 396 397 399 404 410 416 422 426 430 426 414 398 377 359 344 321 289 286 312 326 327 326 324 323 322 322 320 317 312 292 258 220 177 155 153 153 152 151 148 150 159 195 258 298 317 319 304 296 294 310 345 350 324 307 297 269 223 165 93 55 53 55 61 55 39 8 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -1 33 73 104 125 139 144 164 199 202 174 151 133 136 161 191 226 244 243 217 165 139 138 165 220 287 366 395 376 361 350 368 415 457 492 478 415 365 328 365 477 534 538 546 558 540 492 468 468 485 519 563 615 729 903 996 1007 970 886 837 822 822 835 809 742 698 677 683 716 745 768 750 689 654 645 584 470 392 351 311 271 262 284 288 273 260 248 242 240 230 210 203 209 209 204 210 227 240 248 284 346 381 389 394 395 396 397 398 401 406 414 426 440 443 436 416 383 364 359 351 341 340 348 353 353 353 352 352 351 349 346 339 329 311 285 246 196 171 171 171 170 169 168 167 167 188 231 260 277 288 293 292 284 288 304 289 242 211 195 174 149 105 42 17 32 40 40 25 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -1 33 78 116 147 161 158 174 209 214 190 162 131 140 188 240 299 309 270 218 154 121 120 149 208 271 338 368 361 355 350 365 400 415 409 385 345 327 332 392 508 585 623 657 687 694 679 678 693 707 722 728 726 792 928 1013 1048 1044 1002 993 1017 1023 1011 981 932 923 954 996 1051 1091 1119 1102 1041 1011 1011 946 815 722 667 601 525 459 403 350 299 269 259 261 275 267 238 225 228 237 254 272 290 296 290 305 340 361 367 373 378 385 393 399 403 409 418 430 444 449 445 428 398 380 374 371 371 370 367 363 360 355 348 341 334 326 319 310 299 287 273 242 195 171 171 170 168 166 162 157 151 160 185 206 221 235 247 249 241 240 247 231 194 164 143 117 89 52 7 -2 26 50 72 67 37 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -1 32 68 108 121 107 102 106 108 109 93 59 93 194 285 365 356 258 199 179 169 168 182 210 227 233 257 302 327 334 332 321 300 270 254 252 276 325 369 407 470 556 627 681 748 828 894 944 973 982 946 864 852 909 979 1062 1101 1095 1128 1198 1235 1239 1241 1242 1258 1287 1326 1375 1409 1428 1424 1396 1395 1419 1381 1283 1205 1148 1070 971 818 612 458 357 301 290 298 327 323 288 260 240 256 308 346 368 362 327 306 299 303 318 332 344 363 388 405 416 425 433 439 443 445 444 436 421 406 390 382 381 377 368 359 348 333 313 292 272 255 243 233 227 224 226 209 174 156 153 150 146 141 133 124 111 111 123 136 151 161 167 169 168 170 174 177 177 164 138 97 40 4 -2 -2 33 84 152 180 169 123 42 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -1 27 61 72 59 49 40 44 62 64 52 99 208 288 341 317 216 169 178 192 210 224 234 233 223 241 288 320 337 366 406 443 477 477 442 457 522 577 623 683 758 804 821 870 951 1003 1027 1031 1018 976 907 885 912 956 1017 1050 1053 1092 1167 1214 1232 1259 1294 1320 1339 1373 1424 1462 1489 1507 1517 1538 1570 1558 1502 1454 1415 1350 1260 1086 830 637 506 415 363 347 366 360 329 300 272 284 335 370 391 380 337 303 278 269 277 290 309 335 366 389 404 418 432 438 439 439 439 436 428 416 399 390 387 381 372 359 345 323 293 272 261 246 226 214 211 212 218 209 185 170 161 152 140 131 122 111 98 94 98 102 106 108 108 110 115 114 107 108 115 107 82 44 -2 -2 -3 -2 33 82 147 174 163 126 63 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -1 5 12 14 13 9 20 46 75 105 155 227 252 232 196 144 130 152 192 248 276 277 286 303 313 316 332 359 464 646 829 1013 1036 900 858 912 1005 1137 1207 1214 1178 1099 1056 1049 1013 951 893 839 828 858 893 935 942 916 896 883 896 934 971 1005 1048 1099 1124 1123 1153 1212 1266 1314 1364 1415 1454 1480 1490 1486 1482 1479 1452 1400 1269 1058 881 740 605 478 407 392 377 363 344 323 320 335 349 362 353 321 295 275 257 241 247 272 300 329 352 369 390 415 429 431 433 433 429 420 411 402 395 390 384 377 366 350 324 288 280 299 294 265 250 249 248 249 241 227 211 194 173 150 134 127 119 110 107 109 102 85 75 69 72 81 73 45 25 12 -2 -2 -2 -3 -3 -3 -2 25 46 62 56 28 2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -1 -1 4 13 16 35 70 106 143 178 210 205 163 135 120 133 175 222 273 297 292 320 380 424 454 490 532 643 823 978 1107 1127 1038 1038 1128 1225 1330 1355 1301 1219 1110 1076 1117 1105 1039 971 902 876 894 911 925 913 874 856 858 863 870 882 897 925 967 999 1023 1068 1134 1181 1211 1248 1293 1330 1359 1397 1443 1478 1503 1517 1519 1448 1302 1150 992 820 634 502 423 379 369 352 328 324 342 358 373 363 329 300 276 252 228 224 239 258 280 300 319 343 372 385 384 383 383 384 386 390 396 397 392 387 380 369 355 331 299 296 322 320 290 273 271 269 268 264 256 239 214 187 160 143 134 125 115 109 108 96 72 54 43 39 44 29 -2 -2 -2 -3 -3 -3 -3 -3 -3 -3 -2 -2 -2 -2 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -1 3 27 46 60 88 130 156 167 169 160 148 133 130 140 178 243 280 287 288 282 338 455 574 694 786 847 899 940 902 783 773 872 1007 1178 1248 1216 1146 1038 945 868 939 1159 1274 1283 1257 1193 1112 1013 937 884 869 890 924 971 985 967 939 901 885 892 941 1031 1111 1181 1202 1174 1157 1150 1168 1211 1279 1374 1444 1489 1548 1619 1623 1561 1441 1262 1058 830 631 460 366 351 325 289 299 356 396 420 408 359 316 281 254 235 220 210 210 220 235 255 278 304 310 298 292 293 304 326 352 381 395 396 391 381 370 359 344 324 320 331 324 299 283 277 275 277 277 273 255 221 194 172 155 144 129 113 103 98 86 66 47 28 13 3 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 -2 -2 -2 -2 -2 -2 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -1 1 19 42 61 76 101 136 154 155 147 128 122 130 155 198 238 273 281 262 262 283 379 550 733 926 1040 1074 1016 865 690 491 494 697 914 1145 1239 1196 1104 961 860 800 895 1146 1324 1430 1460 1414 1365 1311 1249 1178 1113 1053 1046 1090 1092 1050 1006 958 923 900 938 1036 1124 1200 1223 1192 1158 1119 1128 1185 1251 1327 1397 1459 1536 1627 1643 1584 1478 1325 1152 959 749 522 389 350 309 268 275 332 371 394 381 332 292 259 237 225 208 186 175 176 185 202 222 245 256 253 257 267 287 316 348 384 401 402 397 389 380 372 362 349 343 342 332 314 302 295 292 292 292 292 275 239 210 188 169 152 133 111 94 84 69 47 24 1 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 29 34 41 50 65 86 97 99 55 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -1 21 39 52 60 65 75 89 100 108 112 114 128 154 208 292 311 267 229 198 219 292 441 665 901 1150 1255 1217 1001 608 351 230 284 514 764 1032 1200 1267 1220 1061 954 898 941 1081 1262 1485 1587 1569 1634 1782 1834 1791 1631 1356 1218 1217 1184 1121 1081 1064 1033 986 988 1040 1107 1190 1242 1262 1246 1195 1204 1275 1307 1302 1338 1414 1485 1551 1516 1382 1274 1192 1109 1026 859 609 444 364 304 263 253 272 287 297 285 252 227 211 202 201 189 166 153 148 150 160 175 197 221 248 275 303 329 352 377 404 415 410 407 404 399 392 383 373 363 353 343 335 328 324 318 311 309 314 299 264 233 207 183 161 136 108 85 67 44 16 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 29 102 154 199 238 271 298 327 361 336 253 138 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -1 8 25 39 49 53 51 50 51 57 70 100 146 198 253 300 336 321 254 209 185 215 297 445 658 883 1121 1181 1062 797 384 168 149 252 479 674 837 943 990 955 836 780 785 822 889 1064 1346 1507 1547 1682 1912 2043 2074 1938 1633 1447 1379 1294 1192 1140 1136 1113 1073 1052 1050 1085 1157 1208 1237 1244 1228 1247 1298 1286 1209 1209 1287 1347 1389 1345 1215 1130 1090 1060 1040 897 631 466 403 359 336 323 322 315 301 284 262 248 242 231 215 197 178 165 160 157 157 163 175 201 241 279 316 347 373 397 419 431 431 430 427 422 413 402 389 377 366 357 350 347 346 340 328 322 322 308 280 251 221 193 168 139 107 77 48 17 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 43 119 186 244 292 332 364 400 442 435 378 277 132 20 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -1 8 17 15 21 34 39 34 28 21 25 41 107 222 328 425 427 334 270 236 220 221 247 297 393 535 688 853 831 623 410 192 135 240 390 583 642 565 479 383 324 302 351 472 547 577 736 1024 1231 1357 1519 1718 1893 2045 2042 1886 1730 1576 1421 1265 1183 1173 1165 1161 1128 1066 1057 1100 1121 1121 1155 1223 1258 1259 1189 1049 1014 1084 1128 1147 1131 1080 1042 1017 1005 1006 868 590 456 464 473 481 482 476 450 403 372 359 351 347 319 266 232 218 211 211 204 191 183 181 197 233 270 307 343 377 407 431 450 463 466 459 449 434 417 398 385 380 372 361 358 363 358 344 330 317 302 286 262 229 200 174 143 109 70 26 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 22 81 134 180 220 257 291 323 353 361 347 312 255 198 140 60 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -1 7 12 3 4 16 20 17 11 2 13 44 126 258 373 471 446 299 215 195 194 210 233 264 319 400 483 568 506 296 159 94 133 277 388 467 432 283 143 12 -1 -1 -1 -1 -1 -1 54 262 464 661 874 1102 1306 1485 1556 1520 1454 1359 1278 1210 1172 1162 1153 1146 1130 1106 1094 1094 1086 1069 1113 1217 1272 1277 1195 1027 961 997 1006 988 988 1004 1004 989 957 908 772 548 456 494 525 550 557 546 519 476 451 444 430 411 378 332 319 338 340 327 296 247 216 204 209 232 261 297 328 356 381 405 427 448 455 450 443 432 418 400 392 395 391 380 376 377 369 350 331 313 300 291 272 243 213 183 149 111 65 12 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 11 59 103 141 171 202 233 259 280 293 300 300 293 267 222 144 32 -2 -3 -3 -3 -3 -3 -3 -3 -2 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -1 -1 -1 -1 -1 -1 -1 -1 -1 22 78 156 255 336 398 363 232 155 132 132 155 176 198 224 254 269 268 203 75 36 86 161 260 253 141 54 -1 -1 -1 -3 -3 -3 -3 -3 -3 -1 -1 -1 -1 -1 84 300 419 507 563 643 747 877 1033 1108 1105 1080 1033 1063 1168 1193 1137 1100 1080 1117 1211 1289 1349 1299 1136 1043 1019 975 910 912 982 1012 1003 915 751 614 505 465 493 520 547 552 535 525 522 519 517 489 434 409 414 456 533 550 506 431 324 262 245 237 238 254 286 304 309 322 342 364 387 400 403 406 408 405 395 398 413 416 407 398 388 371 347 327 311 301 295 282 261 232 195 155 112 61 4 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 9 52 93 124 146 167 189 207 222 232 239 244 247 231 195 131 38 -2 -3 -3 -3 -3 -3 -3 -2 5 173 166 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 15 69 127 188 235 270 241 148 95 82 93 128 180 250 275 256 209 133 57 -1 -1 34 91 144 94 -1 -1 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -1 -1 -1 -1 -1 236 500 791 915 872 823 769 837 1026 1109 1086 1084 1104 1145 1207 1265 1322 1308 1225 1166 1131 1076 1002 1006 1090 1135 1141 1049 859 707 590 525 511 512 530 529 510 509 525 530 524 490 430 412 437 494 584 599 540 455 345 284 272 258 242 249 276 293 300 315 339 362 385 396 394 395 397 399 400 407 421 424 417 407 392 374 353 333 315 302 295 285 273 247 208 166 122 72 16 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 16 57 97 126 144 160 177 190 200 208 214 220 228 217 187 138 68 7 -2 -3 -3 -3 -3 -2 54 207 389 385 195 44 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 19 41 58 74 90 81 48 36 45 76 129 242 414 466 398 295 158 61 4 -1 -1 -1 -1 -1 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 146 488 596 470 391 359 459 691 852 943 1039 1142 1196 1203 1204 1198 1229 1295 1329 1329 1305 1256 1261 1322 1367 1397 1349 1223 1039 798 634 547 502 500 490 472 471 488 486 467 438 399 387 403 439 495 494 435 374 311 281 284 273 247 245 268 295 327 359 392 419 438 439 422 408 398 400 414 421 421 418 412 402 389 377 367 350 324 305 292 283 279 259 221 182 142 96 47 14 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 32 75 116 146 164 180 194 205 214 220 222 227 234 225 199 165 121 75 28 -2 -2 -2 -2 51 188 343 515 533 398 225 14 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -1 -1 -1 -1 -1 -1 14 46 86 133 227 367 405 340 261 167 90 30 -1 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -1 -1 -1 16 124 257 416 563 697 891 1147 1249 1197 1169 1165 1227 1355 1428 1445 1441 1416 1423 1462 1484 1491 1480 1452 1300 1022 814 676 592 562 541 529 532 551 544 511 470 421 406 424 447 475 465 418 389 380 378 382 353 290 274 306 342 383 417 445 465 479 475 454 439 431 430 437 434 422 410 398 389 382 376 372 356 330 308 291 282 282 265 233 200 165 127 85 54 33 8 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 13 54 110 151 176 195 210 223 235 244 249 253 256 246 223 197 166 133 98 68 42 43 70 149 281 423 575 606 514 355 131 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -1 -1 28 85 123 141 139 118 102 92 111 161 142 57 -1 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 56 224 200 242 351 644 1121 1302 1188 1160 1219 1300 1406 1466 1481 1488 1486 1495 1513 1491 1427 1448 1554 1492 1262 1063 894 777 711 678 677 689 712 700 652 584 495 466 498 516 521 510 484 497 547 569 562 495 370 334 387 432 468 489 496 502 508 504 491 488 496 490 469 447 423 399 377 368 372 372 367 354 332 312 292 281 279 266 243 219 193 164 131 97 63 32 3 -2 -2 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 -2 81 142 180 207 225 244 264 280 292 296 292 280 259 233 204 180 164 145 122 127 157 226 333 449 573 605 545 436 278 124 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 6 29 40 61 92 101 87 38 -1 -1 -1 6 79 31 -1 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 80 49 71 145 418 891 1108 1069 1101 1204 1291 1362 1405 1421 1443 1472 1497 1519 1484 1393 1402 1510 1499 1371 1211 1020 878 785 739 739 753 780 771 724 650 549 519 561 575 561 551 545 567 616 632 617 551 434 398 441 475 500 509 503 503 510 513 510 515 526 515 482 449 419 391 368 359 365 363 351 338 321 306 291 280 274 261 241 222 202 179 152 120 83 53 30 15 8 4 2 -2 -2 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 53 124 183 222 240 261 286 306 321 327 324 316 304 283 254 234 222 211 200 209 239 294 373 457 546 573 538 462 346 204 37 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 18 112 153 141 113 68 22 -1 -1 -3 -3 -3 -1 -1 -1 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -1 44 72 213 465 676 844 994 1124 1201 1226 1248 1266 1309 1375 1432 1481 1466 1388 1342 1326 1330 1353 1262 1056 898 788 728 719 729 759 760 730 671 584 565 613 624 597 589 602 602 590 573 553 523 485 466 468 473 481 479 467 469 486 501 513 521 524 509 475 442 409 386 372 364 362 349 325 308 297 290 287 279 266 249 228 209 192 173 150 124 92 71 60 58 67 69 66 56 40 23 4 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 25 98 186 238 255 276 301 321 337 347 351 354 357 345 317 293 273 267 275 291 316 352 400 449 497 512 495 436 337 217 77 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -1 -1 -1 -1 41 130 158 126 78 14 -1 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -1 85 204 366 572 755 917 1014 1049 1081 1111 1168 1251 1314 1357 1363 1331 1283 1218 1215 1274 1235 1096 969 852 781 754 747 758 747 715 668 608 600 644 656 634 629 643 632 596 568 547 539 543 536 518 507 501 485 459 454 470 483 494 497 494 480 455 428 401 381 368 359 352 336 311 294 286 280 276 266 249 233 218 204 191 175 156 135 111 96 90 92 102 107 105 98 83 65 42 23 6 -2 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 54 127 210 261 280 300 323 343 362 378 391 404 417 416 400 380 356 349 360 374 392 415 443 465 481 486 481 442 370 273 150 39 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 24 44 31 26 30 42 62 48 -1 -1 -1 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 32 100 174 253 391 588 735 832 905 956 1021 1101 1145 1153 1179 1224 1225 1181 1151 1137 1133 1140 1087 975 894 845 805 776 735 680 643 621 625 656 671 671 671 672 659 634 614 598 596 609 607 590 575 559 527 478 455 460 459 452 444 437 429 421 409 394 376 358 344 335 323 308 296 287 275 259 242 223 212 210 205 198 186 168 153 140 130 121 116 116 118 121 120 114 99 77 62 54 44 31 12 -2 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 -2 45 138 207 254 289 312 334 353 373 394 418 445 466 483 495 502 494 470 458 456 459 467 481 501 507 497 493 496 480 444 369 255 146 44 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -1 28 32 3 -1 -1 -1 -1 -1 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -1 24 34 133 323 515 710 828 870 942 1042 1077 1048 1063 1121 1132 1095 1057 1017 1028 1090 1102 1065 1034 1010 963 893 822 750 708 697 702 723 737 746 732 697 670 652 637 623 623 636 635 622 607 589 554 501 472 467 462 456 447 435 429 429 415 389 365 343 329 321 312 302 292 283 267 245 227 212 206 210 210 206 197 183 174 168 162 155 148 143 141 142 140 135 123 105 94 88 80 69 53 30 14 6 -2 -2 -2 -2 -3 -3 -3 -3 -3 -3 -2 -2 4 43 105 190 248 278 308 338 367 395 418 436 456 479 504 530 549 562 557 533 522 523 520 512 519 542 541 516 502 499 488 470 413 318 227 140 58 -2 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 1 -1 -1 -1 -1 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -1 -1 120 354 680 847 853 927 1069 1106 1040 1011 1020 1004 964 934 914 920 952 1019 1123 1199 1247 1216 1106 1007 919 861 835 829 843 852 857 812 719 665 651 638 625 621 626 624 615 604 592 566 528 502 489 490 506 505 487 477 476 446 387 347 326 313 308 302 293 284 272 256 234 221 216 215 219 219 215 209 202 197 195 193 191 188 184 177 168 158 147 137 129 120 109 102 97 89 78 73 74 64 43 24 5 -2 -3 -3 -3 -3 -2 4 61 105 135 171 213 250 283 318 357 400 448 477 486 493 496 518 558 579 581 571 547 544 562 556 526 529 565 568 539 513 491 470 450 407 341 282 230 187 152 96 19 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 8 215 540 723 764 881 1075 1136 1064 1018 998 972 941 911 883 875 886 977 1148 1285 1387 1406 1342 1269 1187 1105 1023 972 953 933 912 849 744 683 668 654 641 634 631 626 617 608 598 583 565 547 530 526 536 531 510 495 485 448 383 346 335 321 303 290 282 274 265 250 229 219 219 224 231 235 234 232 227 226 227 228 228 227 223 215 203 189 176 166 161 150 135 125 120 114 105 103 106 96 71 46 21 -2 -3 -3 -3 -3 -2 20 91 143 175 201 220 247 282 318 355 399 452 484 495 502 505 527 568 590 593 586 568 571 592 584 546 540 567 565 534 506 480 462 452 420 368 316 263 239 241 223 185 129 54 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 98 292 461 605 805 1061 1165 1118 1080 1050 1031 1022 986 922 889 890 974 1143 1296 1433 1535 1601 1606 1551 1435 1259 1132 1054 982 914 844 772 725 703 686 671 660 652 641 629 618 607 605 610 604 588 569 548 527 505 483 458 423 378 361 371 353 306 278 269 263 260 248 228 220 223 232 247 257 263 264 260 260 263 266 267 265 262 256 245 234 220 209 200 186 166 151 139 127 113 106 104 94 77 54 27 5 -2 -3 -3 -3 -2 6 68 121 164 195 211 238 274 306 333 367 408 441 464 485 505 530 560 582 596 601 597 602 615 605 569 551 549 534 504 482 467 464 474 452 399 329 242 215 249 278 301 264 168 86 18 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 88 227 400 628 910 1075 1123 1144 1136 1132 1132 1090 1006 949 920 979 1127 1284 1449 1589 1703 1753 1738 1669 1547 1420 1286 1151 1014 912 845 794 761 734 713 695 680 665 649 635 625 624 633 628 609 582 547 524 511 492 467 442 417 408 417 398 349 309 278 262 259 249 230 222 224 235 253 270 284 291 291 293 297 300 302 302 300 295 285 273 260 246 234 217 196 177 160 145 130 119 110 97 79 57 30 11 -2 -3 -3 -3 -3 -2 48 102 157 194 213 241 278 309 334 363 396 428 459 491 522 551 576 595 607 616 620 623 626 612 582 561 551 534 508 492 486 484 486 464 418 355 274 255 297 335 369 343 259 184 117 32 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 18 151 353 626 869 1081 1210 1255 1274 1268 1222 1135 1053 975 992 1103 1251 1437 1572 1655 1716 1754 1810 1886 1830 1643 1436 1210 1051 960 890 840 798 764 737 717 696 676 661 650 642 634 618 594 566 535 522 527 523 511 502 497 488 474 455 431 382 308 268 262 251 233 223 222 232 251 272 296 312 322 327 329 331 333 335 337 332 321 307 293 278 261 244 225 204 183 167 156 142 125 103 78 54 32 15 3 -2 -3 -3 -3 -2 32 88 153 198 224 255 293 327 357 386 414 445 480 517 557 589 614 627 627 631 638 635 623 606 584 573 573 564 545 536 537 521 487 455 425 394 360 357 384 395 390 368 329 267 182 77 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -1 164 421 709 1030 1240 1339 1390 1394 1363 1296 1204 1087 1055 1107 1211 1366 1475 1539 1611 1692 1812 1970 1986 1860 1684 1457 1262 1100 981 905 848 809 776 750 726 706 689 675 660 644 622 593 566 543 534 540 543 544 547 552 540 509 493 492 445 354 296 272 254 241 232 226 230 246 269 300 324 342 352 356 360 364 367 368 363 351 338 324 307 288 269 250 227 201 181 170 153 131 107 80 59 44 30 18 6 -2 -3 -3 -2 32 87 157 207 236 268 303 334 363 390 414 448 493 536 576 608 631 634 617 610 613 616 619 612 594 588 596 594 582 575 573 546 494 455 429 410 398 393 395 379 344 319 304 258 183 86 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 58 292 596 970 1234 1390 1482 1511 1513 1488 1401 1254 1167 1140 1163 1237 1302 1357 1442 1556 1678 1807 1895 1942 1894 1753 1543 1263 1068 957 884 847 813 780 755 737 719 700 681 663 637 605 582 570 560 551 553 565 575 583 565 524 513 532 500 417 347 289 258 255 247 234 231 239 260 297 327 352 369 378 386 394 396 394 387 378 366 352 335 313 292 273 246 213 189 172 152 129 107 86 72 66 55 42 26 7 -2 -2 10 47 100 169 219 249 279 308 333 353 374 396 438 500 548 580 606 627 617 577 554 546 565 612 628 611 609 619 623 621 611 595 560 507 463 430 405 390 366 334 290 234 199 187 161 122 61 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 155 454 785 1023 1166 1288 1388 1472 1540 1511 1386 1280 1195 1156 1165 1181 1204 1286 1424 1553 1670 1794 1924 1958 1897 1718 1420 1188 1023 916 866 824 789 764 749 732 715 699 683 664 641 622 607 589 569 566 580 592 602 586 546 533 546 527 475 411 334 286 269 254 242 236 236 256 295 329 357 376 385 394 404 407 403 399 395 386 372 353 330 310 294 270 236 207 181 155 130 112 104 96 88 78 66 57 51 47 44 59 92 138 195 238 265 293 320 342 359 378 399 440 501 545 571 593 609 598 559 531 512 527 575 582 548 535 542 542 533 519 497 464 418 380 349 327 313 281 230 170 103 77 93 99 94 51 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 12 285 481 612 677 818 1035 1248 1458 1537 1485 1396 1270 1188 1148 1112 1079 1140 1296 1436 1558 1683 1809 1880 1893 1791 1572 1342 1103 944 868 812 778 754 741 730 722 714 704 700 701 685 653 623 594 582 589 598 610 603 576 553 534 526 528 488 405 337 284 254 249 244 239 256 295 329 358 375 378 385 396 400 398 399 404 398 382 363 339 323 315 297 269 236 197 162 133 123 132 128 112 100 92 100 125 134 127 138 168 199 234 262 284 309 338 362 382 402 422 453 496 529 551 568 580 577 561 539 511 500 507 476 407 371 367 352 325 302 285 261 230 207 190 178 170 138 82 21 -2 -2 22 70 98 55 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -1 210 374 441 552 708 906 1146 1283 1317 1300 1232 1197 1193 1151 1070 1101 1244 1371 1482 1591 1698 1762 1781 1739 1637 1448 1174 993 907 844 806 789 793 794 790 783 772 765 762 744 710 678 646 626 617 613 615 610 598 572 532 520 535 508 438 373 312 273 258 250 250 268 301 329 352 367 375 384 396 403 407 413 423 417 393 367 336 326 336 326 297 265 229 199 173 168 182 181 164 151 143 156 192 207 199 206 226 245 262 280 299 323 352 377 396 413 428 445 461 483 509 529 544 546 534 505 458 417 383 318 223 175 173 155 121 110 123 118 95 73 52 37 27 -2 -2 -2 -3 -2 39 106 124 73 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 303 449 484 407 449 611 758 891 1000 1085 1185 1299 1296 1175 1164 1265 1357 1440 1518 1590 1605 1563 1568 1618 1508 1237 1062 983 920 872 867 903 920 916 904 885 858 825 797 777 754 727 697 664 637 616 608 612 589 540 510 498 473 434 394 353 310 266 253 271 291 314 329 339 354 375 392 404 416 428 440 453 441 404 364 322 319 355 356 321 295 279 263 248 245 254 254 245 233 218 225 254 266 260 261 268 275 281 293 311 335 363 385 400 412 419 415 397 407 444 476 502 503 479 430 355 280 203 110 -2 -2 -2 -2 -2 -2 9 32 11 -2 -2 -2 -2 -3 -3 -3 -3 -2 142 205 170 106 12 -2 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 227 319 226 215 288 396 541 734 975 1185 1363 1362 1181 1125 1193 1254 1306 1362 1422 1434 1399 1435 1543 1515 1350 1213 1102 1016 954 933 954 967 973 959 926 886 838 812 807 791 765 730 688 654 629 621 629 605 548 507 484 464 447 425 398 351 283 262 286 307 326 336 338 349 371 388 402 418 435 452 469 453 404 350 292 278 308 312 291 290 311 317 309 307 312 310 301 287 268 269 289 297 293 292 295 296 298 311 336 360 382 400 415 426 436 414 361 350 382 422 470 463 403 337 263 196 136 72 4 -2 -2 -2 -2 -3 -2 -2 -2 -3 -3 -3 -3 -3 -3 -3 -2 67 226 276 219 155 85 46 40 1 -2 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 61 161 197 171 193 266 502 902 1197 1386 1351 1092 986 1032 1064 1082 1126 1196 1249 1286 1341 1414 1471 1513 1444 1263 1132 1050 988 946 938 963 950 899 850 804 787 800 792 762 727 688 663 652 647 649 618 555 513 492 479 474 464 447 395 309 275 295 316 338 348 348 353 362 374 390 408 428 449 472 453 393 324 247 205 196 197 208 251 326 361 356 355 358 351 332 314 295 290 297 300 298 300 306 310 313 334 375 399 409 421 438 457 476 441 350 311 322 367 446 426 308 226 181 164 177 200 233 244 234 168 45 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 48 179 291 320 269 219 171 137 118 58 -2 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 16 174 194 185 149 367 838 1143 1280 1214 946 830 866 891 906 959 1052 1146 1240 1305 1340 1440 1604 1592 1404 1253 1141 1057 1001 980 993 966 901 847 806 789 797 790 769 740 702 682 679 673 662 629 575 537 517 504 498 486 470 418 330 292 303 324 357 372 369 369 372 380 395 408 418 421 419 385 321 258 197 153 126 122 142 195 279 333 357 376 392 389 369 350 332 322 320 312 300 300 313 322 329 353 393 415 419 432 453 468 477 439 356 335 376 410 436 389 268 199 180 188 223 273 338 346 296 209 85 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 48 167 238 254 214 183 161 139 118 65 -2 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 145 353 367 186 325 784 1024 1047 953 743 657 695 734 774 858 987 1121 1259 1326 1320 1420 1625 1660 1524 1379 1225 1138 1118 1092 1060 1007 932 877 843 816 798 788 786 767 731 712 712 697 669 639 605 579 559 538 517 493 468 420 349 313 309 333 383 406 400 398 398 405 418 418 405 369 311 251 190 153 140 121 97 87 93 120 170 234 313 372 413 426 412 396 380 366 355 333 300 294 315 333 347 366 392 408 415 432 458 460 437 410 378 421 541 546 439 351 283 252 258 265 274 294 324 285 175 88 22 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 35 72 79 57 51 59 56 42 25 3 -2 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 38 331 405 260 404 836 999 891 751 577 512 556 621 707 812 937 1086 1260 1354 1369 1473 1664 1714 1623 1506 1364 1304 1327 1291 1196 1099 998 927 886 850 821 806 806 797 777 765 762 741 702 667 634 606 582 561 543 513 471 421 361 327 318 342 398 422 413 407 406 407 411 401 378 336 273 224 187 174 186 174 139 115 103 110 136 191 275 342 392 414 410 400 384 367 348 325 297 295 318 336 349 361 371 379 386 409 448 445 399 361 332 382 509 519 410 341 310 293 291 288 284 285 291 233 111 33 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 -2 -2 -2 -2 -2 -2 -2 -2 -2 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 133 303 369 599 993 1063 810 604 445 393 447 550 703 820 900 1041 1243 1391 1485 1594 1720 1756 1701 1634 1556 1553 1626 1575 1399 1240 1097 996 935 892 865 844 830 828 839 840 831 804 760 712 661 619 586 571 575 545 481 421 365 334 328 351 403 422 408 399 394 386 375 359 339 320 304 300 308 318 330 307 249 203 170 161 176 204 243 285 329 356 366 363 347 326 300 288 289 301 321 334 337 337 331 329 331 363 424 423 361 294 221 219 289 333 352 356 347 322 281 258 254 247 238 190 103 43 12 -2 -2 -2 -2 -3 -3 -3 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 48 262 400 633 962 996 736 553 446 439 532 671 856 970 1012 1144 1367 1538 1659 1760 1841 1852 1793 1746 1712 1743 1840 1780 1565 1373 1207 1088 1017 968 942 917 892 891 913 911 887 849 797 742 682 632 591 575 584 551 477 425 396 369 345 353 395 408 393 380 369 358 347 337 329 332 346 367 393 405 402 364 291 234 191 176 187 203 224 245 267 283 293 293 284 270 251 257 288 317 343 347 329 308 284 271 266 290 341 335 273 215 161 164 223 289 360 385 363 319 252 216 210 198 179 144 93 55 29 17 18 14 4 -2 -2 -2 21 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 75 279 354 510 748 801 668 594 576 647 807 979 1163 1260 1268 1390 1626 1792 1889 1967 2025 2003 1899 1843 1834 1876 1971 1910 1695 1500 1326 1203 1128 1078 1050 1022 992 983 996 978 931 877 815 756 698 645 597 573 571 534 460 433 452 431 368 349 374 380 368 352 333 324 327 335 348 369 401 425 443 436 403 347 268 208 168 155 170 191 216 221 205 195 192 192 196 199 201 233 292 342 381 374 322 275 231 204 192 191 201 182 134 124 153 215 310 383 433 425 360 287 206 162 154 137 114 96 83 68 50 44 51 55 55 45 25 6 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 95 302 397 541 732 808 768 764 797 903 1084 1214 1293 1342 1360 1507 1785 1965 2047 2110 2154 2116 1996 1941 1951 2014 2130 2097 1913 1699 1454 1344 1367 1323 1212 1141 1111 1101 1113 1080 1003 921 836 763 703 648 595 566 559 521 452 439 483 473 408 374 369 360 348 332 314 308 316 328 342 367 401 423 433 415 368 309 236 183 150 152 189 220 243 233 190 159 143 142 158 177 199 235 285 329 368 356 296 241 192 164 157 166 193 179 125 133 202 288 389 439 435 389 300 220 149 105 89 67 42 43 70 78 66 58 56 50 42 27 4 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 108 331 528 721 910 1013 1031 1062 1106 1206 1363 1378 1251 1222 1292 1500 1846 2057 2134 2190 2228 2192 2084 2041 2063 2155 2318 2338 2218 1968 1590 1511 1730 1702 1426 1274 1247 1244 1263 1216 1102 982 858 764 700 641 586 554 546 512 452 444 488 495 464 425 380 349 331 319 312 310 314 315 314 325 349 363 365 344 299 250 195 158 138 166 243 289 304 281 219 174 144 143 171 204 242 263 266 280 304 294 249 206 165 150 160 213 311 321 242 238 308 383 462 457 369 279 186 121 81 46 16 -2 -2 -2 56 86 77 59 32 1 -2 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 141 400 683 917 1102 1206 1231 1264 1304 1373 1470 1420 1224 1152 1205 1412 1774 2004 2102 2153 2156 2101 1987 1970 2049 2191 2397 2480 2439 2188 1728 1626 1882 1860 1562 1398 1368 1349 1342 1283 1172 1044 898 795 734 674 615 575 553 516 462 449 479 489 480 451 402 365 339 326 326 326 328 321 306 305 319 325 323 304 270 226 175 146 140 180 266 309 309 283 229 187 156 159 196 237 280 280 238 220 226 213 183 157 135 147 194 278 400 411 312 295 362 411 442 396 272 168 84 30 6 -2 -2 -3 -3 -2 26 57 40 13 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 194 507 863 1130 1307 1387 1370 1373 1395 1408 1409 1344 1211 1130 1099 1246 1572 1808 1956 2001 1943 1846 1709 1731 1911 2125 2372 2523 2579 2360 1867 1690 1828 1806 1623 1512 1472 1418 1350 1282 1213 1105 956 856 804 746 683 627 580 531 481 456 456 457 458 451 437 410 371 353 355 356 358 345 318 306 310 310 306 296 278 237 173 146 157 194 260 281 260 239 219 198 176 189 235 276 313 288 202 150 133 115 97 92 101 156 258 359 459 452 336 307 364 373 333 257 145 57 -2 -2 -2 -3 -3 -3 -3 -3 -2 -2 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 302 635 992 1252 1414 1490 1481 1477 1479 1462 1427 1366 1280 1216 1176 1253 1449 1617 1757 1801 1749 1678 1590 1651 1864 2093 2338 2501 2580 2409 1986 1803 1859 1838 1740 1662 1605 1518 1402 1315 1257 1164 1037 947 895 833 761 694 630 571 516 478 457 444 438 437 442 424 383 363 363 361 357 342 317 303 298 294 291 285 277 239 171 145 162 197 250 266 245 231 224 210 190 199 237 269 294 271 202 167 168 161 147 146 156 216 326 413 478 455 345 307 340 319 243 156 57 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 120 465 782 1070 1284 1424 1518 1565 1577 1554 1535 1520 1484 1427 1409 1431 1429 1404 1429 1505 1553 1573 1597 1625 1728 1905 2095 2297 2415 2447 2337 2084 1963 1974 1956 1911 1847 1766 1649 1496 1380 1301 1222 1141 1069 1006 933 851 774 704 635 567 516 481 449 419 409 420 408 376 357 351 341 326 314 304 294 285 279 276 272 266 231 167 142 157 189 238 263 264 257 243 223 197 191 205 216 224 231 236 269 328 348 330 313 299 327 397 440 456 422 339 295 290 250 174 93 9 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 105 577 923 1143 1307 1413 1511 1599 1638 1628 1602 1561 1533 1518 1543 1608 1601 1521 1488 1501 1509 1512 1539 1590 1717 1918 2093 2242 2314 2309 2246 2124 2065 2066 2071 2079 2033 1932 1785 1590 1444 1344 1262 1197 1131 1064 988 903 824 749 676 603 545 500 455 409 391 401 394 371 351 334 317 298 286 280 272 262 257 255 252 248 221 173 157 171 198 239 262 269 262 243 223 201 195 203 212 221 250 298 360 437 464 440 413 382 387 429 437 411 365 299 252 222 176 113 51 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 640 1060 1211 1319 1382 1471 1584 1661 1700 1663 1551 1515 1554 1619 1710 1769 1797 1787 1739 1665 1564 1505 1487 1620 1903 2088 2173 2199 2166 2135 2107 2108 2137 2183 2245 2219 2104 1926 1685 1504 1384 1285 1207 1135 1069 998 920 843 768 695 625 564 513 461 407 383 387 382 368 345 312 288 274 260 246 237 231 228 230 227 222 209 190 188 203 225 253 264 259 245 223 209 203 210 231 256 284 328 386 443 497 509 480 446 406 398 423 404 343 284 227 177 135 97 62 28 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 638 1054 1093 1186 1333 1464 1578 1648 1673 1630 1521 1500 1567 1648 1743 1834 1921 1926 1849 1727 1559 1468 1454 1607 1928 2131 2215 2206 2102 2057 2072 2115 2188 2282 2397 2386 2248 2029 1727 1518 1401 1299 1213 1134 1060 985 910 835 763 694 628 569 515 461 407 376 368 361 356 337 302 277 262 245 225 213 210 209 211 212 211 207 201 207 222 240 261 267 261 250 234 229 234 246 265 290 320 364 421 470 511 508 460 415 374 355 359 325 254 194 146 102 62 32 14 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 572 910 793 912 1267 1490 1580 1599 1548 1506 1471 1489 1557 1631 1709 1799 1899 1911 1834 1696 1496 1427 1489 1677 1992 2221 2366 2331 2115 2010 2017 2087 2219 2369 2536 2536 2367 2095 1720 1486 1393 1303 1216 1127 1035 950 872 800 736 674 615 558 505 455 408 371 344 332 335 326 303 283 265 242 215 200 198 198 201 206 213 214 209 214 228 244 261 271 273 274 274 280 292 301 306 315 329 359 402 443 481 460 380 322 286 259 239 201 143 95 59 28 2 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 373 742 594 698 1057 1339 1546 1604 1515 1460 1442 1491 1606 1700 1774 1851 1932 1943 1887 1748 1529 1458 1538 1730 2037 2239 2338 2283 2075 1982 2004 2109 2298 2492 2690 2692 2498 2197 1788 1537 1444 1347 1246 1143 1038 943 857 780 712 648 590 536 487 443 404 367 331 314 315 307 289 273 258 238 211 195 190 190 193 200 209 212 207 212 229 246 264 274 277 283 291 300 309 313 312 320 336 376 439 465 454 402 311 247 212 178 146 109 68 33 4 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 42 551 494 545 705 1016 1478 1662 1569 1493 1433 1506 1711 1854 1935 1989 2018 2023 2005 1882 1654 1560 1600 1768 2065 2186 2133 2066 1984 1972 2030 2181 2425 2650 2858 2855 2641 2332 1928 1668 1552 1430 1304 1183 1069 963 865 775 690 617 554 502 460 425 395 364 331 308 297 280 258 246 244 232 212 197 187 184 188 193 199 199 194 203 226 247 267 276 274 277 286 288 283 282 285 304 340 415 528 533 429 336 253 191 149 112 78 50 29 6 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 244 339 399 423 731 1322 1641 1686 1653 1542 1589 1796 1937 2014 2040 2015 2015 2038 1948 1745 1671 1726 1873 2112 2142 1964 1879 1889 1944 2045 2232 2503 2736 2930 2917 2697 2382 1970 1699 1569 1441 1315 1194 1079 970 868 772 683 606 543 490 448 417 396 368 333 306 287 267 246 237 238 231 217 202 185 177 178 179 181 180 176 188 214 236 253 262 263 269 278 276 264 265 280 328 410 506 615 587 420 292 201 135 94 61 34 15 1 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 130 259 208 483 1081 1542 1865 1940 1765 1739 1860 1952 2015 2007 1927 1921 1989 1950 1802 1790 1916 2045 2177 2105 1828 1723 1790 1900 2051 2262 2534 2749 2908 2881 2670 2347 1914 1631 1499 1382 1282 1177 1068 964 865 774 689 616 555 500 452 420 405 378 337 307 286 268 253 243 240 234 227 211 186 169 162 158 156 155 153 167 195 213 223 234 246 257 267 265 251 261 296 390 544 648 700 626 426 270 156 82 47 25 15 2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 157 124 333 786 1287 1836 2061 1959 1894 1865 1880 1937 1914 1812 1794 1860 1856 1782 1835 2015 2112 2125 1996 1725 1641 1742 1892 2091 2326 2597 2781 2879 2837 2655 2349 1920 1635 1494 1374 1274 1168 1055 952 859 773 692 620 556 498 446 413 400 373 330 301 286 277 273 271 270 266 258 238 204 177 156 144 142 141 140 154 182 203 219 235 254 269 280 286 285 317 381 484 625 686 668 563 373 225 119 51 21 5 2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 94 166 281 439 880 1604 2019 2125 2056 1813 1721 1781 1765 1673 1635 1653 1670 1688 1806 2026 2076 1957 1817 1656 1631 1741 1920 2167 2424 2691 2831 2844 2784 2653 2386 1985 1707 1554 1415 1291 1166 1039 934 850 769 691 617 546 484 432 397 381 353 312 290 285 291 308 320 328 325 310 281 240 199 160 139 138 137 137 150 175 205 240 266 285 302 318 339 365 431 535 609 652 622 521 400 260 157 90 43 16 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 64 260 271 580 1185 1684 2076 2081 1699 1514 1527 1524 1504 1457 1381 1415 1558 1736 1950 1979 1824 1724 1678 1696 1779 1914 2103 2316 2553 2683 2707 2688 2627 2415 2050 1784 1617 1457 1305 1163 1030 924 844 763 682 602 522 456 405 370 351 323 287 277 293 313 336 346 343 330 308 273 227 184 145 128 133 138 144 157 178 212 258 295 321 344 363 399 453 535 647 679 630 538 402 285 188 116 70 36 14 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 270 279 384 584 1062 1818 1973 1525 1260 1178 1194 1308 1258 1046 1091 1392 1625 1789 1823 1727 1716 1791 1836 1852 1874 1903 2007 2187 2341 2469 2548 2578 2434 2116 1866 1683 1500 1316 1159 1029 923 840 754 665 575 483 414 366 331 309 283 256 264 309 340 359 351 315 283 253 214 165 132 114 112 128 144 160 176 191 223 274 320 362 394 415 466 548 631 717 693 561 434 310 217 154 102 61 31 13 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 93 240 303 282 683 1505 1718 1322 1060 933 966 1158 1134 894 956 1321 1576 1721 1767 1714 1732 1822 1878 1900 1901 1881 1923 2026 2140 2266 2368 2446 2338 2044 1808 1629 1456 1288 1146 1029 927 838 746 651 558 468 400 352 314 286 260 235 248 299 335 353 339 290 245 203 161 121 100 98 108 129 153 180 200 213 248 306 351 385 436 507 570 626 658 664 595 452 332 237 168 124 84 49 25 11 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 155 334 273 540 1137 1320 1089 913 791 839 1056 1083 921 1008 1344 1589 1743 1808 1783 1771 1773 1823 1921 1992 2033 2058 2065 2077 2095 2147 2232 2129 1838 1614 1458 1326 1220 1122 1031 936 838 739 639 551 476 413 362 318 283 251 223 228 266 296 318 309 268 217 156 115 93 87 98 114 135 165 202 229 245 287 354 388 389 472 637 710 689 616 490 385 301 234 182 136 97 63 36 17 7 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 274 310 516 894 1042 958 872 784 801 924 932 825 924 1228 1465 1635 1735 1764 1773 1760 1815 1937 2041 2126 2139 2078 2023 1974 1985 2056 1954 1680 1484 1367 1269 1193 1111 1024 921 801 692 595 515 455 402 355 313 275 245 222 225 254 273 282 266 225 175 117 83 74 80 101 121 139 168 208 238 259 297 354 386 390 470 624 669 604 495 343 245 199 161 131 100 69 44 23 9 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 124 392 608 774 880 928 937 909 851 765 684 609 706 976 1207 1398 1549 1660 1738 1783 1853 1947 2050 2162 2168 2067 1978 1901 1880 1916 1813 1571 1418 1355 1284 1204 1112 1009 881 729 608 519 451 406 366 331 297 264 242 231 238 264 267 245 210 161 119 84 66 65 78 106 127 139 163 199 229 254 280 308 344 390 432 471 451 374 298 223 172 144 114 83 59 42 26 12 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 92 355 489 609 712 801 876 815 620 500 455 538 747 926 1075 1229 1388 1529 1651 1785 1929 2058 2171 2165 2039 1921 1808 1781 1838 1738 1480 1342 1323 1271 1186 1090 984 849 686 562 476 414 373 338 308 277 245 225 214 216 230 222 192 156 112 81 63 55 57 74 106 124 127 144 175 205 235 259 279 312 358 369 345 299 230 179 148 121 100 76 49 30 19 7 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -1 44 228 315 469 688 695 490 380 363 417 540 623 666 776 953 1150 1368 1613 1885 2066 2155 2131 1995 1850 1696 1687 1822 1729 1408 1256 1272 1233 1140 1045 948 824 672 553 467 401 356 317 284 252 220 194 173 159 151 139 123 103 79 62 53 48 49 67 101 113 104 112 138 168 201 235 269 290 297 283 247 209 169 138 117 93 67 45 27 12 2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -1 20 277 593 631 390 271 272 310 385 439 473 566 718 930 1202 1464 1718 1892 1986 1973 1854 1720 1572 1585 1760 1684 1357 1200 1212 1174 1084 994 903 795 669 565 482 415 362 315 271 232 196 168 148 129 111 96 86 74 61 51 43 43 49 69 102 110 92 92 108 132 164 198 233 247 240 217 179 148 125 104 87 65 40 22 12 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 223 590 622 320 172 180 216 281 372 490 593 682 868 1152 1340 1432 1540 1665 1691 1619 1533 1434 1475 1654 1605 1326 1173 1145 1094 1019 937 848 761 676 596 522 453 391 330 269 217 173 147 140 127 108 92 79 68 57 46 35 38 56 80 109 113 93 83 84 98 124 148 172 185 187 172 140 115 98 79 57 37 18 8 4 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 359 519 345 220 143 134 192 308 481 621 728 915 1182 1320 1328 1386 1494 1517 1456 1398 1344 1389 1534 1503 1296 1152 1071 1003 948 886 817 748 679 611 542 476 412 342 267 206 159 133 128 118 102 88 76 64 53 43 34 42 67 89 108 106 86 74 71 80 100 116 130 138 141 132 112 94 78 59 36 17 3 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 324 465 411 160 63 118 246 446 649 856 1070 1293 1404 1403 1425 1469 1449 1365 1315 1299 1327 1400 1380 1266 1136 989 902 872 841 809 754 678 608 544 483 425 352 263 197 153 126 114 104 95 85 76 64 49 40 39 53 82 97 98 90 72 65 70 79 92 101 106 107 103 99 95 84 65 45 22 5 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 442 515 173 16 42 160 369 596 842 1069 1278 1385 1390 1425 1488 1468 1365 1310 1302 1310 1333 1326 1289 1194 1041 936 880 831 791 740 680 620 561 499 434 352 254 184 142 114 99 88 80 72 65 54 39 33 36 49 73 82 76 66 53 49 56 63 70 74 76 75 71 71 74 68 53 35 14 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 279 533 183 -1 -1 50 251 463 688 913 1140 1266 1291 1386 1550 1573 1455 1382 1354 1337 1331 1340 1364 1325 1223 1105 970 856 763 707 685 647 593 523 438 344 240 167 125 97 82 69 59 49 41 33 25 22 23 29 40 44 42 36 28 26 30 32 34 37 41 44 47 49 49 47 41 29 11 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 366 187 41 -1 -1 134 327 539 748 955 1084 1135 1301 1582 1686 1614 1545 1480 1416 1355 1334 1353 1325 1249 1141 1002 862 719 643 632 604 560 493 403 309 211 143 105 78 62 48 37 28 20 15 12 11 12 15 18 19 17 10 -2 -2 -2 -2 -2 -2 -2 13 20 25 26 28 31 25 10 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 15 185 157 -1 -1 20 190 394 572 723 840 923 1172 1585 1808 1841 1798 1679 1548 1404 1307 1258 1195 1121 1048 979 849 658 549 520 491 463 408 328 248 169 114 81 56 37 24 15 8 3 -2 -2 1 3 5 6 5 2 -2 -3 -3 -3 -3 -3 -3 -3 -2 -2 -2 4 12 23 22 11 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 84 191 27 -1 -1 48 229 385 518 649 778 1084 1568 1869 1988 1970 1817 1636 1430 1304 1258 1198 1122 1080 1071 938 680 514 439 387 360 313 248 186 127 83 55 32 15 3 -2 -2 -2 -3 -3 -2 -2 -2 -2 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 -2 15 20 14 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 146 216 146 -1 -1 43 189 340 510 699 1039 1532 1870 2055 2063 1892 1682 1433 1323 1354 1330 1251 1233 1278 1128 783 536 388 292 250 207 164 123 83 51 26 8 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 9 18 19 4 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 110 285 236 -1 -3 -1 -1 113 352 558 911 1411 1795 2061 2104 1922 1709 1462 1366 1421 1404 1316 1311 1386 1229 838 551 367 249 196 154 125 91 54 21 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 3 16 21 11 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 84 236 219 35 -1 -3 -3 -1 174 357 702 1208 1643 2006 2094 1908 1715 1517 1431 1458 1421 1321 1312 1397 1241 846 557 375 255 196 154 130 91 38 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 12 20 16 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -1 131 100 -1 -3 -3 -3 -1 110 481 929 1378 1830 2030 1979 1857 1663 1541 1490 1412 1308 1290 1359 1239 931 679 483 331 222 153 124 82 28 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 4 14 14 5 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 157 28 -1 -3 -3 -3 -1 248 573 1002 1534 1911 2135 2131 1899 1694 1516 1377 1279 1244 1273 1222 1092 914 690 477 275 152 107 63 21 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 5 8 3 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 89 -1 -3 -3 -3 -3 -1 236 673 1208 1694 2130 2249 2048 1836 1614 1443 1322 1265 1272 1277 1281 1184 986 715 370 168 109 60 20 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 2 1 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 18 -1 -3 -3 -3 -3 -1 392 853 1378 1967 2211 2111 1969 1785 1607 1437 1352 1353 1402 1498 1488 1372 1045 505 201 130 72 24 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 29 -1 -3 -3 -3 -3 -3 -1 457 1009 1655 2029 2129 2095 1927 1763 1602 1513 1496 1550 1678 1713 1654 1307 672 298 187 96 26 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -3 -3 -3 -3 -3 -3 -1 21 587 1197 1702 2101 2213 2040 1909 1818 1748 1699 1723 1822 1859 1835 1504 869 460 279 134 26 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 101 689 1283 1880 2140 2061 2013 1994 1947 1873 1871 1939 1951 1906 1607 1054 641 368 155 2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 131 771 1469 1876 1991 2075 2129 2110 2020 1993 2029 1989 1871 1618 1229 842 455 160 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 260 961 1467 1779 2001 2133 2171 2113 2086 2090 2000 1815 1553 1216 841 429 133 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 355 914 1426 1791 2009 2130 2155 2152 2123 1985 1737 1414 1017 643 292 76 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 556 1149 1582 1858 2029 2097 2130 2126 2000 1752 1418 998 614 264 67 23 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 391 946 1376 1680 1869 1943 2020 2101 2047 1859 1563 1159 753 344 105 35 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 327 920 1342 1592 1758 1838 1927 2026 2029 1937 1741 1442 1090 684 374 159 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 -2 -2 2 3 1 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 363 1069 1479 1595 1696 1782 1851 1903 1948 1987 1952 1845 1621 1281 872 393 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 1 17 21 15 9 3 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 872 1376 1490 1591 1678 1755 1821 1903 2001 2084 2152 2069 1835 1462 949 402 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 21 47 52 34 19 7 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 335 1034 1279 1443 1527 1639 1780 1894 1980 2137 2366 2436 2347 2143 1823 1242 400 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 10 43 84 90 60 34 12 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 177 723 1072 1225 1371 1511 1620 1696 1878 2167 2311 2313 2256 2141 1619 689 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -2 29 69 119 120 74 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -1 479 774 953 1017 1083 1151 1310 1558 1700 1737 1807 1910 1537 689 108 -2 -2 -2 -2 -2 -2 -2 -2 -2 22 58 101 152 144 76 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -1 91 307 548 814 1041 1228 1368 1460 1615 1836 1607 929 466 217 83 63 58 69 81 92 88 69 69 90 124 173 158 80 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -1 13 681 1069 1175 1311 1478 1680 1918 1827 1408 940 425 165 160 174 207 241 275 256 184 140 125 139 181 164 88 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -1 226 665 966 1127 1385 1738 1824 1642 1239 615 293 274 304 382 466 556 537 411 295 189 163 217 198 107 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -1 333 411 733 1299 1599 1633 1362 787 468 405 447 594 757 934 931 749 532 280 196 278 260 140 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -1 -1 58 456 645 555 188 51 144 298 514 767 1056 1173 1116 912 562 388 391 331 209 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -1 -1 -1 -1 -1 -1 -1 143 498 926 1264 1512 1435 1034 740 553 412 317 178 -2 -2 -2 -2 -2 -2 -2 -2 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -1 510 1009 1432 1498 1207 946 712 566 506 512 586 639 672 627 505 347 153 41 13 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 412 878 1103 1086 1007 868 790 776 806 882 903 871 768 594 421 250 130 60 14 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -1 416 635 741 733 735 745 791 873 913 911 828 665 491 306 168 76 16 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -1 148 311 401 416 470 561 668 790 808 721 559 322 155 60 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -1 -1 -1 87 251 418 587 666 653 533 306 145 50 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -1 162 303 403 464 416 260 135 44 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -1 209 297 310 248 146 3 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 82 152 216 271 175 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 -1 191 310 228 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 236 363 303 58 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 200 329 300 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 306 672 525 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
    -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -1 294 858 855 285 -2 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3 -3
  ]
end


; Copyright 2007 Uri Wilensky.
; See Info tab for full copyright and license.
@#$#@#$#@
GRAPHICS-WINDOW
178
10
662
259
-1
-1
2.0
1
20
1
1
1
0
0
0
1
0
237
0
119
1
1
1
ticks
30.0

BUTTON
10
38
81
80
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
95
38
166
80
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

MONITOR
26
97
151
142
water height
word water-height \" meters\"
3
1
11

@#$#@#$#@
## WHAT IS IT?

This model demonstrates one way to locate a continental divide.  A continental divide separates a continent into two regions based on two bodies of water.  Rain in one region flows into one body of water and rain in the other region flows into the other.

In the example data, the continent is North America and the two bodies of water used to calculate the divide are the Pacific and Atlantic oceans.

## HOW IT WORKS

The model is initialized with an elevation map.  Then both oceans systematically rise, bit by bit.  The two floods run towards each other over the continent and eventually crash.  The continental divide is precisely where the two floods collide.

## HOW TO USE IT

SETUP initializes the model.  Elevations are stored in the patches and they are colored appropriately. Also, the two floods are started off on the coasts.

GO runs the model.  When the floods cannot advance any more with the given height of the water, the water level is raised a little bit.  Eventually, when the whole continent has been flooded and the continental divide has been found, the model stops automatically.

## THINGS TO NOTICE

The two floods move at different rates.

The first 100 meters of flood covers more land than the last 100 meters.  What about in between?

Land that's flooded later isn't necessarily higher elevation. (Why?)

## THINGS TO TRY

Use the speed slider to slow the model down and watch what happens in more detail.

Increase the patch-size to get a better view of the action.  (Because the elevation data assumes specific dimensions, you can't change the number of patches in the model.)

## EXTENDING THE MODEL

Make a slider to control how much water-height changes when the flooding at a given water-height has stopped.

Make a slider for controlling how many colors from FLOODED-GROUND-COLOR-LIST get used.  With a smaller number, the flooded land's elevation is easier to see.  With a larger number, the progression of flooding is easier to see.

Is there a difference if `neighbors` is used instead of `neighbors4`? Make a switch to toggle between the two options and compare them.

Try the model with a more detailed dataset.

Allow the user of the model to specify different bodies of water than the Atlantic and Pacific oceans.  For example, it'd be interesting to see which water flows into the Gulf of Mexico and which flows into the Atlantic.

Allow the user to import maps of other parts of the world.

## NETLOGO FEATURES

Note the use of turtles to represent the edges of the flood.  Instead of asking all the patches to find the ones on each edge, we only need to ask the turtles to act.  Since at any given moment only a few patches are at a flood edge, this is much faster.

Note the used of `foreach` on multiple lists to initialize the elevation data in the patches.

## RELATED MODELS

Grand Canyon

## CREDITS AND REFERENCES

This model was inspired by Brian Hayes' article "Dividing the Continent" in American Scientist, Volume 88, Number 6, page 481.  An online version can be found here: https://www.jstor.org/stable/27858114

Thanks to Josh Unterman for his work on this model.

## HOW TO CITE

If you mention this model or the NetLogo software in a publication, we ask that you include the citations below.

For the model itself:

* Wilensky, U. (2007).  NetLogo Continental Divide model.  http://ccl.northwestern.edu/netlogo/models/ContinentalDivide.  Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

Please cite the NetLogo software as:

* Wilensky, U. (1999). NetLogo. http://ccl.northwestern.edu/netlogo/. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

## COPYRIGHT AND LICENSE

Copyright 2007 Uri Wilensky.

![CC BY-NC-SA 3.0](http://ccl.northwestern.edu/images/creativecommons/byncsa.png)

This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 3.0 License.  To view a copy of this license, visit https://creativecommons.org/licenses/by-nc-sa/3.0/ or send a letter to Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.

Commercial licenses are also available. To inquire about commercial licenses, please contact Uri Wilensky at uri@northwestern.edu.

<!-- 2007 -->
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.2.2
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
